<?php

namespace App\Http\Controllers\Api;

use App\Http\Controllers\Controller;
use App\Models\ClientModel;
use App\Models\ProductModel;
use App\Models\QuotesItemsModel;
use App\Models\QuotesModel;
use App\Models\User;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;
use Illuminate\Support\Facades\Log;
use Str;

class MobQuoteController extends Controller
{
    public function index(Request $request)
    {
        $query = QuotesModel::with([
            'client' => function ($query) {
                $query->select('id', 'user_id');
            },
            'client.user' => function ($query) {
                $query->select('id', 'name', 'phone');
            },
            'quotesItem'
        ])->orderBy('id', 'desc')->where('branch_id',$request->branch_id);

        if ($request->has('search') && !empty($request->search)) {
            $searchTerm = $request->search;
            $query->where(function ($q) use ($searchTerm) {
                $q->where('quote_id', 'like', "%$searchTerm%")
                    ->orWhereHas('client.user', function ($userQuery) use ($searchTerm) {
                        $userQuery->where('name', 'like', "%$searchTerm%")
                            ->orWhere('phone', 'like', "%$searchTerm%");
                    });
            });
        }

        $perPage = $request->per_page ?? 10;
        $quotes = $query->paginate($perPage);
        return response()->json($quotes);
    }

    public function store(Request $request)
    {
       

        $request->validate([
            'products.*.qty' => 'required|numeric',
            'products.*.unit_price' => 'required|numeric',
            'products.*.amount' => 'required|numeric',
            'client_num' => 'required',
        ]);

        DB::beginTransaction();

        try {
            $user = User::where('phone', $request->client_num)->first();

            $client_id = $this->saveClient($request, $user);

            if (!$client_id) {
                return response()->json([
                    'success' => false,
                    'message' => 'Failed to create client'
                ], 400);
            }

            $lastQuotes = QuotesModel::orderBy('id', 'desc')->first();
            $quote_id = $this->generateQuoteId($lastQuotes);

            $quote = QuotesModel::create([
                'quote_id' => $quote_id,
                'client_id' => $client_id,
                'quote_date' => $request->quote_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type ?? null,
                'discount' => $request->discount ?? null,
                'note' => $request->note,
                'term' => $request->term,
                'branch_id' => $request->branch_id,
                'status' => $request->status ?? '',
            ]);

            foreach ($request->products as $field) {
                $product = ProductModel::find($field['id']);

                QuotesItemsModel::create([
                    'quote_id' => $quote->id,
                    'product_id' => $field['id'],
                    'product_name' => $product->name ?? '',
                    'available_quantity' => $product->qty,
                    'quantity' => $field['qty'],
                    'price' => $field['unit_price'],
                    'total' => $field['amount'],
                ]);
            }

            DB::commit();

            return response()->json([
                'success' => true,
                'message' => 'Quote created successfully',
                'data' => [
                    'quote_id' => $quote->id,
                    'quote_number' => $quote->quote_id
                ]
            ]);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Error creating quote: ' . $e->getMessage());

            return response()->json([
                'success' => false,
                'message' => 'Failed to create quote',
                'error' => $e->getMessage()
            ], 500);
        }
    }

    public function update(Request $request)
    {
        if ($request->type == 'convert') {
            return $this->processConvert($request);
        }

        // $request->validate([
        //     'products.*.qty' => 'required|numeric|lte:products.*.available_qty',
        //     'products.*.unit_price' => 'required|numeric',
        //     'products.*.amount' => 'required|numeric',
        // ]);


        DB::beginTransaction();

        try {
            $quote = QuotesModel::findOrFail($request->id);

            $quote->update([
                'quote_date' => $request->quote_date,
                'due_date' => $request->due_date,
                'amount' => $request->sub_total,
                'final_amount' => $request->total_amount,
                'discount_type' => $request->discount_type ?? null,
                'discount' => $request->discount ?? null,
            ]);

            $productIds = [];

            foreach ($request->products as $field) {
                $product = ProductModel::find($field['id']);

                QuotesItemsModel::updateOrCreate(
                    [
                        'quote_id' => $quote->id,
                        'product_id' => $field['id'],
                    ],
                    [
                        'product_name' => $product->name ?? '',
                        'available_quantity' => $product->qty,
                        'quantity' => $field['qty'],
                        'price' => $field['unit_price'],
                        'total' => $field['amount'],
                    ]
                );

                $productIds[] = $field['id'];
            }

            // Remove deleted items
            QuotesItemsModel::where('quote_id', $quote->id)
                ->whereNotIn('product_id', $productIds)
                ->delete();

            DB::commit();

            return response()->json([
                'status' => true,
                'message' => 'Quote updated successfully',
                'data' => $quote
            ], 200);

        } catch (\Exception $e) {
            DB::rollBack();

            \Log::error('Error updating quote: ' . $e->getMessage());

            return response()->json([
                'status' => false,
                'message' => 'Failed to update quote',
                'error' => $e->getMessage(),
            ], 500);
        }
    }

    public function destroy($id)
    {
        DB::beginTransaction();
        try {
            $quote = QuotesModel::with('quotesItem.quoteItemTaxes')->findOrFail($id);

            $quote->quotesItem->each(function ($item) {
                $item->quoteItemTaxes()->delete();
            });

            $quote->quotesItem()->delete();
            $quote->delete();

            DB::commit();

            return response()->json(['message' => 'Quote deleted successfully'], 204);

        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('While Deleting Quote' . $e->getMessage());
            return response()->json(['message' => 'An error occurred while deleting the quote'], 500);
        }
    }

    protected function saveClient(Request $request, $user = null)
    {
        try {
            if ($user) {
                $client = ClientModel::where('user_id', $user->id)->first();
                return $client->id;
            }

            $randomEmail = Str::random(10) . '@billing.com';
            $user = User::create([
                'name' => $request->client_name,
                'email' => $randomEmail,
                'phone' => $request->client_num,
                'password' => Hash::make('client@123'),
                'user_flg' => 'C',
                'active_status' => 'A',
                'branch_id' => $request->branch_id
            ]);

            $client = ClientModel::create([
                'user_id' => $user->id,
                'branch_id' => $request->branch_id,
                'active_status' => 'A',
            ]);

            return $client->id;
        } catch (\Exception $e) {
            \Log::error('While create Client:' . $e);
            return null;
        }
    }

    protected function generateQuoteId($lastQuotes)
    {
        if ($lastQuotes) {
            $lastQuotesId = $lastQuotes->quote_id;
            $parts = explode('-', $lastQuotesId);
            $prefix = $parts[0];
            $number = (int) $parts[1];
            $newNumber = str_pad($number + 1, strlen($parts[1]), '0', STR_PAD_LEFT);
            return $prefix . '-' . $newNumber;
        }
        return 'OSQOT-001';
    }

}
